package uk.ac.warwick.util.ais.apim.config;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.inject.AbstractModule;
import com.google.inject.Provides;
import com.google.inject.name.Named;
import com.google.inject.Singleton;
import com.typesafe.config.ConfigException;
import org.apache.http.HttpResponse;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.impl.nio.client.CloseableHttpAsyncClient;
import play.api.Configuration;
import uk.ac.warwick.util.ais.apim.rodb.RodbHttpClient;
import uk.ac.warwick.util.ais.apim.stutalk.StuTalkHttpClient;
import uk.ac.warwick.util.ais.auth.Authenticator;
import uk.ac.warwick.util.ais.auth.credentials.OAuth2ClientCredentials;
import uk.ac.warwick.util.ais.auth.model.ClientCredentialsRequest;
import uk.ac.warwick.util.ais.auth.model.ClientCredentialsResult;
import uk.ac.warwick.util.ais.auth.model.OAuth2TokenFetchParameters;
import uk.ac.warwick.util.ais.auth.token.AccessToken;
import uk.ac.warwick.util.ais.auth.token.TokenCache;
import uk.ac.warwick.util.ais.core.httpclient.AisHttpAsyncClient;
import uk.ac.warwick.util.ais.core.httpclient.AisHttpResponseHandler;
import uk.ac.warwick.util.ais.core.httpclient.HttpRequestBuilder;
import uk.ac.warwick.util.ais.core.httpclient.HttpRequestExecutor;
import uk.ac.warwick.util.ais.core.json.AisJsonConverter;
import uk.ac.warwick.util.ais.core.properties.AisApimProperties;

import java.util.function.Function;

/**
 * this is a Guice module for Play Application.
 * to use this module, add the following to conf file:
 * <pre>
 * {@code
 * play.modules.enabled += uk.ac.warwick.util.ais.apim.config.AisPlayModule
 * }
 * </pre>
 */
public class AisPlayModule extends AbstractModule {

    private final AisSpringConfiguration configTemplate = new AisSpringConfiguration();

    @Override
    protected void configure() {
        // no-op
    }

    @Named("aisApacheHttpAsyncClient")
    @Provides
    @Singleton
    public CloseableHttpAsyncClient provideHttpAsyncClient(Configuration configuration) {
        Integer maxConnections = getConfigIntOrDefault(configuration, "ais.apim.http.maxConnections", 10);
        return configTemplate.httpAsyncClient(maxConnections);
    }

    @Named("stuTalkObjectMapper")
    @Provides
    @Singleton
    public ObjectMapper provideStuTalkObjectMapper(Configuration configuration) {
        boolean emptyStringToNull = getConfigBooleanOrDefault(configuration, "ais.apim.objectMapper.emptyStringToNull", true);
        // only apply new line Deserializer and Serializer when property is set.
        boolean normaliseNewline = getConfigBooleanOrDefault(configuration, "ais.apim.objectMapper.normaliseNewline", false);
        return configTemplate.stuTalkObjectMapper(emptyStringToNull, normaliseNewline);
    }

    @Named("stuTalkJsonConverter")
    @Provides
    @Singleton
    public AisJsonConverter provideStuTalkJsonConverter(
            @Named("stuTalkObjectMapper") ObjectMapper objectMapper) {
        return configTemplate.stuTalkJsonConverter(objectMapper);
    }

    @Named("aisHttpResponseHandler")
    @Provides
    @Singleton
    public AisHttpResponseHandler<HttpResponse> provideAisHttpResponseHandler(
            @Named("stuTalkJsonConverter") AisJsonConverter stuTalkJsonConverter) {
        return configTemplate.aisHttpResponseHandler(stuTalkJsonConverter);
    }

    @Named("aisHttpRequestExecutor")
    @Provides
    @Singleton
    public HttpRequestExecutor<HttpUriRequest, HttpResponse> provideAisHttpRequestExecutor(
            @Named("aisApacheHttpAsyncClient") CloseableHttpAsyncClient httpAsyncClient) {
        return configTemplate.aisHttpRequestExecutor(httpAsyncClient);
    }

    @Named("aisOAuth2ClientCredentials")
    @Provides
    @Singleton
    public OAuth2ClientCredentials provideOAuth2ClientCredentials(Configuration configuration) {
        String authority = getConfigString(configuration, "ais.apim.auth.oauth2.auth_url");
        String clientId = getConfigString(configuration, "ais.apim.auth.oauth2.client_id");
        String clientSecret = getConfigString(configuration, "ais.apim.auth.oauth2.client_secret");

        return configTemplate.aisOAuth2ClientCredentials(authority, clientId, clientSecret);
    }

    @Named("aisAuthTokenCache")
    @Provides
    @Singleton
    public TokenCache provideAisAuthTokenCache() {
        return configTemplate.aisAuthTokenCache();
    }

    @Named("aisAuthTokenFetcher")
    @Provides
    @Singleton
    public Function<OAuth2TokenFetchParameters, AccessToken> provideAisAuthTokenFetcher(
            @Named("stuTalkObjectMapper") ObjectMapper objectMapper,
            @Named("aisHttpRequestExecutor") HttpRequestExecutor<HttpUriRequest, HttpResponse> httpRequestExecutor) {

        return configTemplate.aisAuthTokenFetcher(objectMapper, httpRequestExecutor);
    }

    @Named("aisAuthenticator")
    @Provides
    @Singleton
    public Authenticator<ClientCredentialsRequest, ClientCredentialsResult> provideAisAuthenticator(
            @Named("aisOAuth2ClientCredentials") OAuth2ClientCredentials credentials,
            @Named("aisAuthTokenFetcher") Function<OAuth2TokenFetchParameters, AccessToken> tokenFetcher,
            @Named("aisAuthTokenCache") TokenCache tokenCache) {
        return configTemplate.aisAuthenticator(credentials, tokenFetcher, tokenCache);
    }

    @Named("aisApimProperties")
    @Provides
    @Singleton
    public AisApimProperties provideAisApimProperties(Configuration configuration) {
        String bapiChannelId = getConfigString(configuration, "ais.apim.headers.bapi-channel-id");
        String bapiAppId = getConfigString(configuration, "ais.apim.headers.bapi-app-id");
        String baseUrl = getConfigString(configuration, "ais.apim.base-url");

        return configTemplate.aisApimProperties(bapiChannelId, bapiAppId, baseUrl);
    }

    @Named("aisHttpRequestBuilder")
    @Provides
    @Singleton
    public HttpRequestBuilder<HttpUriRequest> provideAisHttpRequestBuilder(
            @Named("stuTalkJsonConverter") AisJsonConverter jsonConverter,
            @Named("aisApimProperties") AisApimProperties aisApimProperties) {
        return configTemplate.aisHttpRequestBuilder(jsonConverter, aisApimProperties);
    }

    @Named("aisAuthHttpRequestBuilder")
    @Provides
    @Singleton
    public HttpRequestBuilder<HttpUriRequest> provideAisAuthHttpRequestBuilder(
            Configuration configuration,
            @Named("aisAuthenticator") Authenticator<ClientCredentialsRequest, ClientCredentialsResult> authenticator,
            @Named("aisHttpRequestBuilder") HttpRequestBuilder<HttpUriRequest> requestBuilder) {

        String defaultScope = getConfigString(configuration, "ais.apim.auth.oauth2.scope");
        return configTemplate.aisAuthHttpRequestBuilder(defaultScope, authenticator, requestBuilder);
    }

    @Named("aisHttpAsyncClient")
    @Provides
    @Singleton
    public AisHttpAsyncClient provideAisHttpAsyncClient(
            @Named("aisAuthHttpRequestBuilder") HttpRequestBuilder<HttpUriRequest> aisAuthHttpRequestBuilder,
            @Named("aisHttpResponseHandler") AisHttpResponseHandler<HttpResponse> responseHandler,
            @Named("aisHttpRequestExecutor") HttpRequestExecutor<HttpUriRequest, HttpResponse> httpRequestExecutor,
            @Named("stuTalkObjectMapper") ObjectMapper objectMapper) {
        return configTemplate.aisHttpAsyncClient(aisAuthHttpRequestBuilder, responseHandler, httpRequestExecutor, objectMapper);
    }

    @Named("stuTalkHttpClient")
    @Provides
    @Singleton
    public StuTalkHttpClient provideStuTalkHttpClient(
            @Named("aisHttpAsyncClient") AisHttpAsyncClient aisHttpAsyncClient) {
        return configTemplate.stuTalkHttpClient(aisHttpAsyncClient);
    }

    @Named("rodbHttpClient")
    @Provides
    @Singleton
    public RodbHttpClient provideRodbHttpClient(
            @Named("aisHttpAsyncClient") AisHttpAsyncClient aisHttpAsyncClient) {
        return configTemplate.rodbHttpClient(aisHttpAsyncClient);
    }

    private String getConfigString(Configuration configuration, String key) {
        return configuration.underlying().getString(key);
    }

    private Integer getConfigIntOrDefault(Configuration configuration, String key, Integer defaultValue) {
        try {
            return configuration.underlying().getInt(key);
        } catch (ConfigException.Missing e) {
            return defaultValue;
        }
    }

    private boolean getConfigBooleanOrDefault(Configuration configuration, String key, boolean defaultValue) {
        try {
            return configuration.underlying().getBoolean(key);
        } catch (ConfigException.Missing e) {
            return defaultValue;
        }
    }
}
