package uk.ac.warwick.util.ais.core.json;

import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.junit.Test;

import java.io.IOException;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNull;

public class PipelineStringDeserializerTest {

    private final ObjectMapper objectMapper = new ObjectMapper();
    private PipelineStringDeserializer deserializer;
    private final String json = "{\n" +
            "        \"mapc\": \"CH910-10\",\n" +
            "        \"mabs\": \"\",\n" +
            "        \"desc\": \"Neque porro quisquam est qui\\\\n dolorem ipsum quia dolor sit amet,\\n consectetur, adipisci velit...\"\n" +
            "      }";

    @Test
    public void deserialize_emptyStringOnly_returnNull() throws IOException {
        EmptyStringToNullTransformer transformer = new EmptyStringToNullTransformer();
        deserializer = new PipelineStringDeserializer(transformer);

        JsonNode jsonNode = objectMapper.readTree(json);
        JsonParser emptyNode = jsonNode.get("mabs").traverse();
        emptyNode.nextToken();
        JsonParser stringNode = jsonNode.get("desc").traverse();
        stringNode.nextToken();
        assertEquals("Neque porro quisquam est qui\\n dolorem ipsum quia dolor sit amet,\n consectetur, adipisci velit...", deserializer.deserialize(stringNode, null));
        assertNull(deserializer.deserialize(emptyNode, null));
    }

    @Test
    public void deserialize_normalizeNewLineValueOnly_returnConvertedValue() throws IOException {
        StringTransformer transformer = TransformerPipelineFactory.createDeserializer(false, true);
        deserializer = new PipelineStringDeserializer(transformer);

        JsonNode jsonNode = objectMapper.readTree(json);
        JsonParser emptyNode = jsonNode.get("mabs").traverse();
        JsonParser stringNode = jsonNode.get("desc").traverse();
        stringNode.nextToken();
        emptyNode.nextToken();
        assertEquals("Neque porro quisquam est qui\r\n dolorem ipsum quia dolor sit amet,\r\n consectetur, adipisci velit...", deserializer.deserialize(stringNode, null));
        assertEquals("", deserializer.deserialize(emptyNode, null));
    }

    @Test
    public void deserialize_mixedPipeline_returnConvertedValue() throws IOException {
        StringTransformer transformer = TransformerPipelineFactory.createDeserializer(true, true);
        deserializer = new PipelineStringDeserializer(transformer);

        JsonNode jsonNode = objectMapper.readTree(json);
        JsonParser emptyNode = jsonNode.get("mabs").traverse();
        JsonParser stringNode = jsonNode.get("desc").traverse();
        stringNode.nextToken();
        emptyNode.nextToken();
        assertNull(deserializer.deserialize(emptyNode, null));
        assertEquals("Neque porro quisquam est qui\r\n dolorem ipsum quia dolor sit amet,\r\n consectetur, adipisci velit...", deserializer.deserialize(stringNode, null));
    }
}
