package uk.ac.warwick.util.ais.core.json;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.JsonNodeFactory;
import org.junit.Test;
import uk.ac.warwick.util.ais.core.exception.AisJsonProcessingException;

import java.util.HashMap;
import java.util.Map;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;

public class AbstractAisJsonConverterTest {

    private final ObjectMapper objectMapper = new ObjectMapper();
    private final AbstractAisJsonConverter converter = new AbstractAisJsonConverter(objectMapper) {};

    @Test
    public void fromJsonNode_validJsonNodeAndTypeReference_returnsDeserializedObject() {
        TypeReference<String> typeReference = new TypeReference<String>() {};
        JsonNode jsonNode = JsonNodeFactory.instance.textNode("test");
        String result = converter.fromJsonNode(jsonNode, typeReference);

        assertEquals("test", result);
    }

    @Test(expected = IllegalArgumentException.class)
    public void fromJsonNode_nullJsonNode_throwsIllegalArgumentException() {
        converter.fromJsonNode(null, new TypeReference<String>() {});
    }

    @Test(expected = AisJsonProcessingException.class)
    public void fromJsonNode_invalidJsonNode_throwsAisJsonProcessingException() {
        JsonNode jsonNode = JsonNodeFactory.instance.arrayNode();
        converter.fromJsonNode(jsonNode, new TypeReference<String>() {}); // array node cannot be deserialized to a string
    }

    @Test
    public void fromJsonNode_forClass_returnsDeserializedObject() {
        JsonNode jsonNode = JsonNodeFactory.instance.textNode("test");
        String result = converter.fromJsonNode(jsonNode, String.class);
        assertEquals("test", result);
    }

    @Test(expected = IllegalArgumentException.class)
    public void fromJsonNode_forClassWithNullJsonNode_throwsIllegalArgumentException() {
        converter.fromJsonNode(null, String.class);
    }

    @Test
    public void fromJsonString_validJsonStringAndTypeReference_returnsDeserializedObject() {
        String jsonString = "{\"key\":\"value\"}";
        TypeReference<Map<String, String>> typeReference = new TypeReference<Map<String, String>>() {};
        Map<String, String> result = converter.fromJsonString(jsonString, typeReference);

        assertEquals("value", result.get("key"));
    }

    @Test(expected = IllegalArgumentException.class)
    public void fromJsonString_nullJsonString_throwsIllegalArgumentException() {
        converter.fromJsonString(null, new TypeReference<String>() {});
    }

    @Test(expected = AisJsonProcessingException.class)
    public void fromJsonString_invalidJsonString_throwsAisJsonProcessingException()  {
        String jsonString = "{\"key\":value\"}"; // missing double quote
        converter.fromJsonString(jsonString, new TypeReference<String>() {});
    }

    @Test
    public void toJsonString_validObject_returnsJsonString() {
        Map<String, String> object = new HashMap<>();
        object.put("key", "value");
        String result = converter.toJsonString(object);

        assertEquals("{\"key\":\"value\"}", result);
    }

    @Test
    public void toJsonString_nullObject_returnsEmptyJsonObject() {
        String result = converter.toJsonString(null);
        assertEquals("{}", result);
    }

    @Test
    public void toJsonNode_validJsonString_returnsJsonNode() {
        String jsonString = "{\"key\":\"value\"}";
        JsonNode result = converter.toJsonNode(jsonString);

        assertNotNull(result);
        assertEquals("value", result.get("key").asText());
    }

    @Test(expected = IllegalArgumentException.class)
    public void toJsonNode_nullJsonString_throwsIllegalArgumentException() {
        converter.toJsonNode(null);
    }

    @Test(expected = AisJsonProcessingException.class)
    public void toJsonNode_invalidJsonString_throwsAisJsonProcessingException()  {
        String jsonString = "{\"key\":value\"}"; // missing double quote
        converter.toJsonNode(jsonString);
    }
}
