package uk.ac.warwick.util.ais.core.httpclient;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.JsonNodeFactory;
import org.apache.http.HttpResponse;
import org.junit.Before;
import org.junit.Test;
import uk.ac.warwick.util.ais.core.exception.AisHttpException;
import uk.ac.warwick.util.ais.core.json.AisJsonConverter;

import java.io.IOException;

import static org.junit.Assert.assertEquals;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

public class AbstractAisHttpResponseHandlerTest {

    private AisJsonConverter jsonConverter;

    private final AbstractAisHttpResponseHandler<HttpResponse> defaultHandler =
            new AbstractAisHttpResponseHandler<HttpResponse>(jsonConverter) {
                @Override
                protected JsonNode handleResponseEntity(HttpResponse response) throws IOException {
                    return null;
                }
            };

    @Before
    public void setUp() {
        jsonConverter = mock(AisJsonConverter.class);
    }

    @Test(expected = AisHttpException.class)
    public void handleResponse_IOException_throwAisHttpException() {
        AbstractAisHttpResponseHandler<HttpResponse> handler = new AbstractAisHttpResponseHandler<HttpResponse>(jsonConverter) {
            @Override
            protected JsonNode handleResponseEntity(HttpResponse response) throws IOException {
                throw new IOException("Test exception");
            }
        };

        handler.handleResponse(mock(HttpResponse.class), new TypeReference<String>() {});
    }

    @Test
    public void handleResponse_success() {
        JsonNode jsonNode = JsonNodeFactory.instance.textNode("test");
        TypeReference<String> typeReference = new TypeReference<String>() {};
        AbstractAisHttpResponseHandler<HttpResponse> handler = new AbstractAisHttpResponseHandler<HttpResponse>(jsonConverter) {
            @Override
            protected JsonNode handleResponseEntity(HttpResponse response) throws IOException {
                return jsonNode;
            }
        };

        when(jsonConverter.fromJsonNode(jsonNode, typeReference)).thenReturn("test");

        String result = handler.handleResponse(mock(HttpResponse.class), typeReference);
        assertEquals("test", result);
    }

    @Test
    public void handleError_AisHttpException_returnAisHttpException() {

        AisHttpException aisHttpException = new AisHttpException(
                AisHttpException.ErrorType.CONNECTION_ERROR,
                "Network exception occurred.",
                new RuntimeException("Test exception")
        );
        AisHttpException result = defaultHandler.handleError(aisHttpException);
        assertEquals(aisHttpException, result); // Check if the same object is returned
    }

    @Test
    public void handleError_SocketException_returnAisHttpException() {

        AisHttpException result = defaultHandler.handleError(new java.net.SocketException());
        assertEquals(AisHttpException.ErrorType.CONNECTION_ERROR, result.getErrorType());
        assertEquals("(errorType=CONNECTION_ERROR,message=Network exception occurred.)", result.getMessage());
    }

    @Test
    public void handleError_InterruptedIOException_returnAisHttpException() {
        AisHttpException result = defaultHandler.handleError(new java.io.InterruptedIOException());
        assertEquals(AisHttpException.ErrorType.CONNECTION_ERROR, result.getErrorType());
        assertEquals("(errorType=CONNECTION_ERROR,message=Network exception occurred.)", result.getMessage());
    }

    @Test
    public void handleError_OtherException_returnAisHttpException() {

        AisHttpException result = defaultHandler.handleError(new RuntimeException());
        assertEquals(AisHttpException.ErrorType.OTHERS, result.getErrorType());
        assertEquals("(errorType=OTHERS,message=An error occurred while communicating with AIS API Management Service.)", result.getMessage());
    }
}
