package uk.ac.warwick.util.ais.core.helpers;

import org.apache.http.client.methods.HttpUriRequest;
import org.junit.Before;
import org.junit.Test;
import uk.ac.warwick.util.ais.auth.OAuth2ClientCredentialsAuthenticator;
import uk.ac.warwick.util.ais.auth.exception.AuthenticationException;
import uk.ac.warwick.util.ais.auth.model.ClientCredentialsRequest;
import uk.ac.warwick.util.ais.auth.model.ClientCredentialsResult;
import uk.ac.warwick.util.ais.auth.token.AccessToken;
import uk.ac.warwick.util.ais.core.httpclient.AisHttpRequest;
import uk.ac.warwick.util.ais.core.httpclient.HttpRequestBuilder;
import uk.ac.warwick.util.collections.Pair;

import java.util.concurrent.atomic.AtomicBoolean;

import static org.junit.Assert.*;
import static org.mockito.Mockito.*;

public class AisHttpRequestAuthorizerTest {

    private static final String DEFAULT_SCOPE = "default-scope";
    private static final String TEST_METHOD = "GET";
    private static final String AUTHORIZATION_HEADER_KEY = "Authorization";

    private final AisHttpRequest request = new AisHttpRequest.Builder().path("path").build();
    private final ClientCredentialsResult result = new ClientCredentialsResult(
            DEFAULT_SCOPE,
            new AccessToken("test-token", "Bearer", 1800L)
    );

    private OAuth2ClientCredentialsAuthenticator authenticator;

    @Before
    public void setUp() {
        authenticator = mock(OAuth2ClientCredentialsAuthenticator.class);
    }

    @Test(expected = IllegalArgumentException.class)
    public void constructor_DefaultScopeIsNull_ThrowException() {
        HttpRequestBuilder<HttpUriRequest> delegate = (method, req) -> mock(HttpUriRequest.class);
        new AisHttpRequestAuthorizer<>(null, authenticator, delegate);
    }

    @Test(expected = IllegalArgumentException.class)
    public void constructor_DefaultScopeIsEmpty_ThrowException() {
        HttpRequestBuilder<HttpUriRequest> delegate = (method, req) -> mock(HttpUriRequest.class);
        new AisHttpRequestAuthorizer<>("", authenticator, delegate);
    }

    @Test(expected = IllegalArgumentException.class)
    public void constructor_AuthenticatorIsNull_ThrowException() {
        HttpRequestBuilder<HttpUriRequest> delegate = (method, req) -> mock(HttpUriRequest.class);
        new AisHttpRequestAuthorizer<>(DEFAULT_SCOPE, null, delegate);
    }

    @Test(expected = IllegalArgumentException.class)
    public void constructor_DelegateBuilderIsNull_ThrowException() {
        new AisHttpRequestAuthorizer<>(DEFAULT_SCOPE, authenticator, null);
    }

    @Test
    public void buildHttpRequest_shouldAddAuthorizationHeader() {
        when(authenticator.authenticate(any(ClientCredentialsRequest.class))).thenReturn(result);
        HttpUriRequest finalRequest = mock(HttpUriRequest.class);
        AtomicBoolean isDelegateCalled = new AtomicBoolean(false);
        HttpRequestBuilder<HttpUriRequest> delegate = (method, req) -> {
            isDelegateCalled.set(true);
            return finalRequest;
        };

        AisHttpRequestAuthorizer<HttpUriRequest> authorizer = new AisHttpRequestAuthorizer<>(DEFAULT_SCOPE, authenticator, delegate);

        HttpUriRequest httpUriRequest = authorizer.buildHttpRequest(TEST_METHOD, request);

        verify(authenticator).authenticate(any(ClientCredentialsRequest.class));
        Pair<String, String> token = request.getHeaders().stream()
                .filter(x -> x.getLeft().equals(AUTHORIZATION_HEADER_KEY))
                .findFirst().orElse(null);
        assertNotNull(token);
        assertEquals("Bearer test-token", token.getRight());
        assertEquals(finalRequest, httpUriRequest);
        assertTrue(isDelegateCalled.get());
    }

    @Test
    public void buildHttpRequest_authenticationFailed_throwException() {
        when(authenticator.authenticate(any(ClientCredentialsRequest.class)))
                .thenThrow(new AuthenticationException("Authentication failed", null));
        HttpUriRequest finalRequest = mock(HttpUriRequest.class);
        AtomicBoolean isDelegateCalled = new AtomicBoolean(false);
        HttpRequestBuilder<HttpUriRequest> delegate = (method, req) -> {
            isDelegateCalled.set(true);
            return finalRequest;
        };

        AisHttpRequestAuthorizer<HttpUriRequest> authorizer = new AisHttpRequestAuthorizer<>(DEFAULT_SCOPE, authenticator, delegate);

        try {
            authorizer.buildHttpRequest(TEST_METHOD, request);
            fail("Should throw an exception");
        } catch (Exception ex) {
            assertTrue(ex instanceof AuthenticationException);
            verify(authenticator).authenticate(any(ClientCredentialsRequest.class));
            assertFalse(isDelegateCalled.get()); // delegate should not be called
        }
    }

}
