package uk.ac.warwick.util.ais.auth;

import org.junit.Test;
import uk.ac.warwick.util.ais.auth.credentials.OAuth2ClientCredentials;
import uk.ac.warwick.util.ais.auth.exception.AuthenticationException;
import uk.ac.warwick.util.ais.auth.exception.TokenFetchException;
import uk.ac.warwick.util.ais.auth.model.ClientCredentialsRequest;
import uk.ac.warwick.util.ais.auth.model.ClientCredentialsResult;
import uk.ac.warwick.util.ais.auth.model.OAuth2TokenFetchParameters;
import uk.ac.warwick.util.ais.auth.token.AccessToken;
import uk.ac.warwick.util.ais.auth.token.TokenCache;

import java.util.function.Function;

import static org.junit.Assert.*;

public class OAuth2ClientCredentialsAuthenticatorTest {

    private final OAuth2ClientCredentials credentials = new OAuth2ClientCredentials("test_url", "clientId", "clientSecret");
    private final AccessToken newToken = new AccessToken("new_token", "Bearer", 3600L);
    private final Function<OAuth2TokenFetchParameters, AccessToken> tokenFetcher = oAuth2TokenFetchParameters -> newToken;

    @Test
    public void authenticate_validRequestWithCache_returnsCachedToken() throws AuthenticationException {
        TokenCache tokenCache = new TokenCache(); // Create a new cache
        OAuth2ClientCredentialsAuthenticator authenticator = new OAuth2ClientCredentialsAuthenticator(credentials, tokenFetcher, tokenCache);
        AccessToken cachedToken = new AccessToken("cached_token", "Bearer", 1800L);
        tokenCache.getAccessToken("clientId#scope", () -> cachedToken); // Cache the token before testing

        ClientCredentialsRequest request = new ClientCredentialsRequest("scope", false); // Request to use cache
        ClientCredentialsResult result = authenticator.authenticate(request);

        assertEquals("scope", result.getScope());
        assertEquals("cached_token", result.getAccessToken());
        assertEquals("Bearer", result.getTokenType());
        assertFalse(result.isExpired());
        assertFalse(result.isExpiringAfter(1799));
        assertTrue(result.isExpiringAfter(1801));
    }

    @Test
    public void authenticate_validRequestWithoutCache_fetchesNewToken() throws AuthenticationException {
        TokenCache tokenCache = new TokenCache(); // Create a new cache
        OAuth2ClientCredentialsAuthenticator authenticator = new OAuth2ClientCredentialsAuthenticator(credentials, tokenFetcher, tokenCache);

        ClientCredentialsRequest request = new ClientCredentialsRequest("scope", false); // Request to use cache
        ClientCredentialsResult result = authenticator.authenticate(request);

        assertEquals("scope", result.getScope());
        assertEquals("new_token", result.getAccessToken());
        assertEquals("Bearer", result.getTokenType());
        assertFalse(result.isExpired());
        assertFalse(result.isExpiringAfter(3599));
        assertTrue(result.isExpiringAfter(3601));
    }

    @Test(expected = AuthenticationException.class)
    public void authenticate_tokenFetcherThrowsException_throwsAuthenticationException() throws AuthenticationException {
        TokenCache tokenCache = new TokenCache(); // Create a new cache
        Function<OAuth2TokenFetchParameters, AccessToken> failureFetcher = parameters -> {
            throw new TokenFetchException("Failed to fetch token", 500, null);
        };
        OAuth2ClientCredentialsAuthenticator authenticator = new OAuth2ClientCredentialsAuthenticator(credentials, failureFetcher, tokenCache);

        ClientCredentialsRequest request = new ClientCredentialsRequest("scope", false); // Request to use cache
        authenticator.authenticate(request);
    }

    @Test
    public void authenticate_skipCache_fetchesNewToken() throws AuthenticationException {
        TokenCache tokenCache = new TokenCache(); // Create a new cache
        OAuth2ClientCredentialsAuthenticator authenticator = new OAuth2ClientCredentialsAuthenticator(credentials, tokenFetcher, tokenCache);
        AccessToken cachedToken = new AccessToken("cached_token", "Bearer", 1800L);
        tokenCache.getAccessToken("clientId#scope", () -> cachedToken); // Cache the token before testing

        ClientCredentialsRequest request = new ClientCredentialsRequest("scope", true); // skip cache
        ClientCredentialsResult result = authenticator.authenticate(request);

        assertEquals("scope", result.getScope());
        assertEquals("new_token", result.getAccessToken());
        assertEquals("Bearer", result.getTokenType());
        assertFalse(result.isExpired());
        assertFalse(result.isExpiringAfter(3599));
        assertTrue(result.isExpiringAfter(3601));
    }

}
