package uk.ac.warwick.util.ais.core.json;

import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.databind.JsonSerializer;
import com.fasterxml.jackson.databind.SerializerProvider;

import java.io.IOException;

/**
 * A custom Jackson {@link JsonSerializer} for {@link String} values that delegates transformation
 * logic to a configurable {@link StringTransformer} pipeline.
 *
 * <p>This serializer allows for applying one or more transformation steps before writing
 * string values to JSON—for example, normalizing newline characters, trimming whitespace, or formatting strings.</p>
 *
 * @see StringTransformer
 * @see ChainedTransformer
 * @see PipelineStringDeserializer
 */
public class PipelineStringSerializer extends JsonSerializer<String> {

    private final StringTransformer transformer;

    public PipelineStringSerializer(StringTransformer transformer) {
        this.transformer = transformer;
    }

    /**
     * Applies the transformation pipeline to the input string and writes the result to JSON.
     *
     * @param input              the input string to transform and write
     * @param jsonGenerator      the JSON generator
     * @param serializerProvider the serializer provider
     * @throws IOException if writing fails
     */
    @Override
    public void serialize(String input, JsonGenerator jsonGenerator, SerializerProvider serializerProvider) throws IOException {
        jsonGenerator.writeString(transformer.transform(input));
    }
}
