package uk.ac.warwick.util.ais.core.json;

/**
 * A {@link StringTransformer} that normalises newline characters for serialization by converting
 * Windows-style newlines (<code>\r\n</code>) to Unix-style newlines (<code>\n</code>).
 * <p>
 * Background:
 * <ul>
 *   <li>The AIS API converts <code>\r\n</code> into <code>\n\n</code>, which causes layout issues.</li>
 *   <li>To avoid this, only <code>\n</code> is stored in the database and serialized in JSON.</li>
 *   <li>This transformer guarantees that outgoing strings use normalized Unix-style newlines.</li>
 * </ul>
 * </p>
 * <p>To reverse this behavior during deserialization, see {@link NewlineDeserializerTransformer}.</p>
 */
public class NewlineSerializerTransformer implements StringTransformer {

    /**
     * Transforms the input string by converting all <code>\r\n</code> sequences to <code>\n</code>.
     *
     * @param input the input string to normalize
     * @return the string with normalized newlines, or null if input is null
     */
    @Override
    public String transform(String input) {
        if (input == null) return null;
        return input.replace("\r\n", "\n");
    }
}
