package uk.ac.warwick.util.ais.core.json;

import java.util.regex.Pattern;

/**
 * <p>
 * A {@link StringTransformer} that normalises newline characters by converting any newline characters (<code>\n</code>)
 * that are not already preceded by a carriage return (<code>\r</code>) into Windows-style newlines (<code>\r\n</code>).
 * </p>
 * <p>
 * Background:
 * <ul>
 *   <li>The AIS API converts <code>\r\n</code> into <code>\n\n</code>.</li>
 *   <li>To prevent this, we store only <code>\n</code> in the database.</li>
 *   <li>This transformer restores <code>\r\n</code> formatting when reading text (e.g., during deserialization).</li>
 * </ul>
 * </p>
 * <p>
 * Regex explanation:
 * <ul>
 *   <li><code>(?&lt;!\\r|\\\\r)</code> is a negative lookbehind that ensures the <code>\n</code> is not already preceded by a literal <code>\r</code> or an escaped <code>\\r</code>.</li>
 *   <li><code>(\\\\n|\\n)</code> matches both escaped and literal newline characters.</li>
 * </ul>
 * </p>
 * <p>
 * On the flip side, during serialization (see {@link NewlineSerializerTransformer}), <code>\r\n</code> is converted to <code>\n</code>.
 * </p>
 */
public class NewlineDeserializerTransformer implements StringTransformer {
    private static final Pattern NEWLINE_PATTERN = Pattern.compile("(?<!\\r|\\\\r)(\\\\n|\\n)");

    /**
     * Transforms the input string by normalising newline characters.
     * Converts lone <code>\n</code> (not already preceded by <code>\r</code>) to <code>\r\n</code>.
     *
     * @param input the input string to normalize
     * @return the normalized string, or null if input is null
     */
    @Override
    public String transform(String input) {
        if (input == null) return null;
        return NEWLINE_PATTERN.matcher(input).replaceAll("\r\n");
    }
}
