package uk.ac.warwick.util.ais.core.httpclient;

import com.fasterxml.jackson.core.type.TypeReference;
import uk.ac.warwick.util.ais.core.exception.AisHttpException;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;

/**
 * An interface representing a client for sending HTTP requests.
 */
public interface AisHttpAsyncClient {

    /**
     * Sends an HTTP request asynchronously.
     *
     * @param method       the HTTP method to use for the request
     * @param request      the {@link AisHttpRequest} object containing the HTTP request details
     * @param responseType the {@link TypeReference} representing the expected response type
     * @param <T>          the type of the response object
     * @return a {@link CompletableFuture} that will be completed with the response object of type T
     *         when the request completes successfully, or with an exception if the request fails
     */
    <T> CompletableFuture<T> sendRequestAsync(String method, AisHttpRequest request, TypeReference<T> responseType);

    /**
     * Sends an HTTP request synchronously, blocking until a response is received or the timeout expires.
     *
     * @param method       the HTTP method to use for the request
     * @param request      the {@link AisHttpRequest} object containing the HTTP request details
     * @param responseType the {@link TypeReference} representing the expected response type
     * @param timeout      the maximum time to wait for a response
     * @param unit         the {@link TimeUnit} of the timeout parameter
     * @param <T>          the type of the response object
     * @return the response object of type T
     * @throws AisHttpException         if the timeout is exceeded before a response is received
     */
    <T> T sendRequest(String method, AisHttpRequest request, TypeReference<T> responseType, long timeout, TimeUnit unit)
            throws AisHttpException;
}
