package uk.ac.warwick.util.ais.core.helpers;

import uk.ac.warwick.util.ais.auth.Authenticator;
import uk.ac.warwick.util.ais.auth.model.AuthenticationResult;
import uk.ac.warwick.util.ais.auth.model.ClientCredentialsRequest;
import uk.ac.warwick.util.ais.auth.model.ClientCredentialsResult;
import uk.ac.warwick.util.ais.core.httpclient.AisHttpRequest;
import uk.ac.warwick.util.ais.core.httpclient.HttpRequestBuilder;
import uk.ac.warwick.util.collections.Pair;

import static uk.ac.warwick.util.ais.core.helpers.AisApimConstants.AUTHORIZATION_HEADER_KEY;

/**
 * An {@link HttpRequestBuilder} that authenticates with AIS before executing the request.
 */
public class AisHttpRequestAuthorizer<R> implements HttpRequestBuilder<R> {

    protected static final String SPACE = " ";
    protected final String defaultScope;
    protected final Authenticator<ClientCredentialsRequest, ClientCredentialsResult> authenticator;
    protected final HttpRequestBuilder<R> delegate;

    public AisHttpRequestAuthorizer(String defaultScope,
                                    Authenticator<ClientCredentialsRequest, ClientCredentialsResult> authenticator,
                                    HttpRequestBuilder<R> delegate) {
        if (defaultScope == null || defaultScope.isEmpty()) {
            throw new IllegalArgumentException("defaultScope cannot be null or empty.");
        }
        if (authenticator == null) {
            throw new IllegalArgumentException("OAuth2ClientCredentialsAuthenticator cannot be null.");
        }
        if (delegate == null) {
            throw new IllegalArgumentException("HttpRequestBuilder cannot be null.");
        }
        this.defaultScope = defaultScope;
        this.authenticator = authenticator;
        this.delegate = delegate;
    }

    @Override
    public R buildHttpRequest(String method, AisHttpRequest request) {

        addAuthorizationHeader(request); // add the authorization header to the request before actually building it
        return delegate.buildHttpRequest(method, request);
    }

    /**
     * Typically this will authenticate with AIS using a default scope that has access to all protected resources/APIs.
     * If you need to get the authorization header for a specific scope.
     * You can override this in the subclass and call the doAuthentication(scope, skipCache) method.
     *<p>
     * You can also ignore the authorization header if the API does not require it,
     * it may be useful when you need to access public APIs.
     *
     * @param request the request - it might need for your checking which scope to use
     */
    protected void addAuthorizationHeader(AisHttpRequest request) {

        AuthenticationResult result = doAuthentication(defaultScope, false);

        String authorization = result.getTokenType() + SPACE + result.getAccessToken();
        request.addHeader(new Pair<>(AUTHORIZATION_HEADER_KEY, authorization));
    }

    /**
     * However, sometimes we just want to acquire an access token that only has access to one or a few specific resources,
     * so this method is for such cases.
     *
     * @param scope the scope of the access token
     * @param skipCache whether to skip the cache and force a new token to be fetched
     * @return the authorization header value. e.g. "Bearer eyJ0eXAiOiJKV1QiLCJhbGciOiJSUz..."
     */
    protected AuthenticationResult doAuthentication(String scope, boolean skipCache) {
        ClientCredentialsRequest request = new ClientCredentialsRequest.Builder()
                .addScope(scope)
                .skipCache(skipCache)
                .build();
        return authenticator.authenticate(request);
    }
}
