package uk.ac.warwick.util.ais.auth.token;

import java.time.Instant;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Supplier;

/**
 * The TokenCache is responsible for caching the OAuth2 access token in-memory.
 * Thread-safety is ensured using a ReentrantLock to prevent multiple simultaneous refreshes.
 * The token aim to be reused for all HTTP requests until it expires.
 *
 */
public final class TokenCache {

    // The token is about to expire within 5 minutes.
    private static final int TOKEN_EXPIRY_THRESHOLD = 5 * 60;

    private final Map<String, AccessToken> cache = new LinkedHashMap<>();

    private final ReentrantLock lock = new ReentrantLock();

    /**
     * Retrieves the AccessToken for the given key.
     * If the token is null or will expire within 5 minutes, it fetches a new one using the provided TokenSupplier.
     *
     * @param key         the key for which to retrieve the token
     * @param tokenSupplier a supplier that fetches a new token when needed
     * @return the AccessToken associated with the given key
     */
    public AccessToken getAccessToken(String key, Supplier<AccessToken> tokenSupplier) {
        lock.lock();
        try {
            AccessToken token = cache.get(key);

            // Check if the token is null or expiring within 5 minutes
            if (token == null || isExpiringSoon(token)) {
                token = tokenSupplier.get();
                cache.put(key, token);
            }

            return token;
        } finally {
            lock.unlock();
        }
    }

    /**
     * Determines if an AccessToken is expiring within the next 5 minutes.
     *
     * @param token the AccessToken to check
     * @return true if the token is expiring soon, false otherwise
     */
    private boolean isExpiringSoon(AccessToken token) {
        Instant threshold = Instant.now().plusSeconds(TOKEN_EXPIRY_THRESHOLD);
        return token.getExpiryTime().isBefore(threshold);
    }

}
