package uk.ac.warwick.util.ais.auth;

import uk.ac.warwick.util.ais.auth.model.AuthenticationResult;

import java.util.HashMap;
import java.util.Map;

/**
 * AuthenticatorRegistry is a class responsible for storing Authenticators and returning them when needed based on the serviceId.
 * You will register the Authenticator for each service into the registry during initialization.
 */
public class AuthenticatorRegistry {
    private final Map<String, Authenticator<?, ? extends AuthenticationResult>> registry = new HashMap<>();

    /**
     * Register an Authenticator inside the registry
     * @param serviceId The service id that used this authenticator
     * @param authenticator the Authenticator used to authenticate the service
     */
    public <T, R extends AuthenticationResult> void register(String serviceId, Authenticator<T, R> authenticator) {
        registry.put(serviceId, authenticator);
    }

    /**
     * Get Authenticator based on service id
     *
     * @return The Authenticator associated with the given service id,
     * will return null when there are no authenticator associated with a given service id.
     */
    @SuppressWarnings("unchecked")
    public <T, R extends AuthenticationResult> Authenticator<T, R> getAuthenticator(String serviceId) {
        return (Authenticator<T, R>) registry.get(serviceId);
    }
}
